program BRACKETINGMETHODS;
{--------------------------------------------------------------------}
{  Alg2'23.pas   Pascal program for implementing Algorithm 2.2-3     }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 2.2 (Bisection Method).                                 }
{  Section   2.2, Bracketing Methods for Locating a Root, Page 61    }
{                                                                    }
{  Algorithm 2.3 (False position or Regula Falsi Method).            }
{  Section   2.2, Bracketing Methods for Locating a Root, Page 62    }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    MAX = 100;
    FunMax = 9;
    GNmax = 230;
    MaxN = 300;

  type
    AVECTR = array[0..MAX] of real;
    RVECTOR = array[0..GNmax] of real;
    LETTER = string[8];
    LETTERS = string[200];
    Status = (Computing, Done, More, Working);
    DoSome = (Go, Stop);

  var                                               {Program Variables}
    Cond, FunType, Inum, K, KL, KR, Kcount, Meth, Sub: integer;
    DNpts, GNpts, N: integer;
    A, B, C, D, DX, Delta, Epsilon, Rnum: real;
    A0, A1, A2, B0, B1, B2, C0, C1, C2: real;
    Xmax, Xmin, Ymax, Ymin: real;
    YA, YB, YC: real;
    VA, VB, VC: AVECTR;
    Xg, Yg: RVECTOR;
    Satisfied: BOOLEAN;
    Stat, State: STATUS;
    Ans, AnsW: CHAR;
    DoMo: DoSome;
    Mess: LETTERS;

  function F (X: real): real;
  begin
    case FunType of
      1: 
        F := X * SIN(X) - 1;
      2:
        F := EXP(X) - 2 - X;
      3: 
        F := COS(X) + 1 - X;
      4: 
        F := LN(ABS(X)) - 5 + X;
      5: 
        F := X * X - 10 * X + 23;
      6: 
        F := 1 / (X - 2);
      7: 
        F := SIN(X) / COS(X);
      8: 
        F := X * X - 2 * X - 1;
      9: 
        F := X * X * X - X - 3;
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      1:
        WRITELN('f(x) = x sin(x) - 1');
      2: 
        WRITELN('f(x) = exp(x) - 2 - x');
      3: 
        WRITELN('f(x) = cos(x) + 1 - x');
      4: 
        WRITELN('f(x) = ln(x) - 5 + x');
      5: 
        WRITELN('f(x) = x^2 - 10 x + 23');
      6: 
        WRITELN('f(x) = 1/(x-2)');
      7:
        WRITELN('f(x) = tan(x)');
      8: 
        WRITELN('f(x) = x^2 - 2 x - 1');
      9: 
        WRITELN('f(x) = x^3 - x - 3');
    end;
  end;

  procedure BISECT ( {FUNCTION F(X:real): real:}
                  A, B, Delta: real; var C, D: real;
                  var A0, A1, A2, B0, B1, B2, C0, C1, C2: real;
                  var Cond, K, KL, KR: integer);
    const
      Big = 1E5;
    var
      Max: integer;
      YA, YB, YC: real;
  begin
    K := 0;
    KL := 0;
    KR := 0;
    YA := F(A);
    YB := F(B);
    D := B - A;
    Cond := 0;                                     {0:State is iteration}
    Max := 1 + TRUNC((Ln(D) - Ln(Delta)) / Ln(2));
    if YA * YB > 0 then
      Cond := 1;                              {1:Method does NOT apply}
    while (Cond = 0) and (K < MAX) do
      begin
        C := (A + B) / 2.0;
        YC := F(C);
        if K = 0 then
          begin
            A0 := A;
            B0 := B;
            C0 := C;
          end;
        if K = 1 then
          begin
            A1 := A;
            B1 := B;
            C1 := C;
          end;
        if K = 2 then
          begin
            A2 := A;
            B2 := B;
            C2 := C;
          end;
        VA[K] := A;
        VB[K] := B;
        VC[K] := C;
        if YC = 0 then
          begin
            A := C;
            B := C;
            Cond := 2;                         {2:A perfect ZERO was found}
          end
        else
          begin
            if YB * YC > 0 then
              begin                                    {Squeeze from Right}
                B := C;
                YB := YC;
                KR := KR + 1;
              end
            else
              begin
                A := C;                                   {Squeeze from Left}
                YA := YC;
                KL := KL + 1;
              end;
          end;
        K := K + 1;
      end;
    VA[K] := A;
    VB[K] := B;
    VC[K] := (A + B) / 2;
    D := B - A;
    if D < Delta then
      begin
        if Cond <> 2 then
          Cond := 3;                            {3:Root is within Tolerance}
        if (ABS(YA) > Big) and (ABS(YB) > Big) then
          Cond := 4;                            {4:A Pole of f(x) was found}
      end;
  end;                                        {End of PROCEDURE BISECT}

  procedure FALSEPO (A, B: real; var C, DX: real;
                     Delta, Epsilon: real; var K: integer;
                     var A0, A1, A2, B0, B1, B2, C0, C1, C2: real;
                     var Satisfied: BOOLEAN);
    label
      999;
    var
      M, YA, YB, YC: real;
  begin
    YA := F(A);
    YB := F(B);
    K := 0;
    Satisfied := False;
    if YA * YB > 0 then
      goto 999;
    while (K < MAX) and (Satisfied = False) do
      begin
        M := (B - A) / (YB - YA);
        if ABS(YA) < ABS(YB) then
          begin
            DX := YA * M;
            C := A - DX;
          end
        else
          begin
            DX := YB * M;
            C := B - DX;
          end;
        YC := F(C);
        if K = 0 then
          begin
            A0 := A;
            B0 := B;
            C0 := C;
          end;
        if K = 1 then
          begin
            A1 := A;
            B1 := B;
            C1 := C;
          end;
        if K = 2 then
          begin
            A2 := A;
            B2 := B;
            C2 := C;
          end;
        VA[K] := A;
        VB[K] := B;
        VC[K] := C;
        if (C - A) < DX then
          DX := C - A;
        if YC = 0 then
          Satisfied := TRUE
        else
          begin
            if YB * YC > 0 then
              begin
                B := C;
                YB := YC;
              end
            else
              begin
                A := C;
                YA := YC;
              end;
          end;
        if (ABS(DX) < Delta) and (ABS(YC) < Epsilon) then
          Satisfied := True;
        K := K + 1;
      end;
    VA[K] := A;
    VB[K] := B;
    M := (B - A) / (YB - YA);
    if ABS(YA) < ABS(YB) then
      begin
        DX := YA * M;
        VC[K] := A - DX;
      end
    else
      begin
        DX := YB * M;
        VC[K] := B - DX;
      end;
999:
  end;                                       {End of PROCEDURE FALSEPO}

  procedure MESSAGE (var FunType, Meth: integer);
    var
      I, K: integer;
  begin
    CLRSCR;
    for I := 1 to 3 do
      WRITELN;
    WRITELN('                  ROOT FINDING: BRACKETING METHODS');
    WRITELN;
    WRITELN;
    WRITELN('          A bracketing method will be used to find the roots of');
    WRITELN;
    WRITELN('     the equation  f(x) = 0.  Either the bisection method or the ');
    WRITELN;
    WRITELN('     Regula-Falsi (false position method) can be used.');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('            < 1 > Bisection method.');
    WRITELN;
    WRITELN('            < 2 > Regula-Falsi (false position) method.');
    WRITELN;
    WRITELN;
    Mess := '                  SELECT your method  < 1 or 2 > ?  ';
    WRITE(Mess);
    READLN(Meth);
    if Meth < 1 then
      Meth := 1;
    if Meth > 2 then
      Meth := 2;
    CLRSCR;
    WRITELN;
    if Meth = 1 then
      WRITELN('     You chose the bisection method to find roots of  f(x) = 0.')
    else
      WRITELN('     You chose the false position method to find roots of  f(x) = 0.');
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('     <', K : 2, ' >   ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    Mess := '     SELECT your function  < 1 - 9 > ?  ';
    WRITE(Mess);
    READLN(FunType);
    if FunType < 1 then
      FunType := 1;
    if FunType > FunMax then
      FunType := FunType;
  end;

  procedure GETPOINTS (var A, B, Delta, Epsilon: real);
    var
      T: real;
  begin
    CLRSCR;
    if Meth = 1 then
      WRITELN('     You chose the bisection method to find roots of  f(x) = 0.')
    else
      WRITELN('     You chose the false position method to find roots of  f(x) = 0.');
    WRITELN;
    WRITE('     ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('An interval [a,b] must be specified on which f(a) & f(b) have opposite signs.');
    WRITELN;
    WRITELN('A nested sequence of intervals  {[a ,b ]}  are found which bracket the zero.');
    WRITELN('                                   k  k');
    if Meth = 1 then
      begin
        WRITELN('The interval [a ,b ] is one half as wide as [a   ,b   ] for k = 1,2,...,N.');
        WRITELN('               k  k                           k-1  k-1');
        WRITELN;
        WRITELN('Convergence is declared when  |b  - a | < Delta.');
        WRITELN('                                N    N');
        WRITELN;
        WRITELN('The final answer the midpoint value is  c  = (b  - a )/2.');
        WRITELN('                                         N     N    N');
      end
    else
      begin
        WRITELN;
        WRITELN('The values  c  = b  - f(b )*[b  - a ]/[f(b ) - f(a )]  are computed');
        WRITELN('             k    k      k    k    k      k       k');
        WRITELN;
        WRITELN('for k = 1,2,...,N.      Convergence is declared when:');
        WRITELN;
        WRITELN('( |c  - a | < Delta  OR  |b  - c | < Delta )  AND  ( |f(c )| < Epsilon ).');
        WRITELN('    N    N                 N    N                        N');
      end;
    Mess := '     ENTER the left  endpoint  a = ';
    WRITE(Mess);
    READLN(A);
    Mess := '     ENTER the right endpoint  b = ';
    WRITE(Mess);
    READLN(B);
    if B < A then
      begin
        T := A;
        A := B;
        B := T;
      end;
    Mess := '     ENTER the tolerance   Delta = ';
    Delta := 0.000000001;
    WRITE(Mess);
    READLN(Delta);
    Delta := ABS(Delta);
    if (Delta < 0.000000001) then
      Delta := 0.000000001;
    if Meth = 2 then
      begin
        Mess := '     ENTER the tolerance Epsilon = ';
        Epsilon := 0.000000001;
        WRITE(Mess);
        READLN(Epsilon);
        Epsilon := ABS(Epsilon);
        if (Epsilon < 0.000000001) then
          Epsilon := 0.000000001;
      end;
    WRITELN;
    VA[0] := A;
    VB[0] := B;
    VC[0] := (A + B) / 2;
    YA := F(A);
    YB := F(B);
    YC := F(C);
  end;                                     {End of PROCEDURE GETPOINTS}

  procedure RESULTSB (A, B, Delta, C, D: real; A0, A1, A2, B0, B1, B2, C0, C1, C2: real; Cond, K, KL, KR: integer);
    var
      J: integer;
  begin
    CLRSCR;
    WRITELN('The bisection method was used to find a zero of the function');
    WRITELN;
    PRINTFUNCTION(FunType);
    WRITELN;
    if Cond <> 1 then
      begin
        WRITELN('        Left point             Midpoint              Right point');
        WRITELN;
        WRITELN(A0 : 15 : 7, '        ', C0 : 15 : 7, '        ', B0 : 15 : 7);
        if K > 1 then
          WRITELN(A1 : 15 : 7, '        ', C1 : 15 : 7, '        ', B1 : 15 : 7);
        if K > 2 then
          WRITELN(A2 : 15 : 7, '        ', C2 : 15 : 7, '        ', B2 : 15 : 7);
      end;
    WRITELN;
    case Cond of
      1: 
        begin
          WRITELN('The values  f(a)  and  f(b)  do not differ in sign.');
          WRITELN;
          WRITELN('f(', A : 15 : 7, '  ) =', F(A) : 15 : 7);
          WRITELN('f(', B : 15 : 7, '  ) =', F(B) : 15 : 7);
          WRITELN;
          WRITELN('Thus, the bisection method cannot be used.');
        end;
      2, 3: 
        begin
          WRITELN('The bisection method took ', K : 3, ' iterations.');
          WRITELN;
          WRITELN('There were', KR : 3, ' squeezes from the right');
          WRITELN('       and', KL : 3, ' squeezes from the left.');
          WRITELN;
          WRITELN('             The root is    c =', C : 15 : 7);
          WRITELN;
          WRITELN('           The accuracy is   +-', D : 15 : 7);
          WRITELN;
          if D < Delta then
            WRITELN('which is smaller than   Delta =', Delta : 15 : 7)
          else
            WRITELN(' which is LARGER than   Delta =', Delta : 15 : 7);
          WRITELN;
          WRITELN('     The function value is   f(', C : 15 : 7, '  ) =', F(C) : 15 : 7);
          if Cond = 2 then
            begin
              WRITELN;
              WRITELN('The function value is exactly zero!');
            end;
        end;
      0, 4: 
        begin
          WRITELN('The current iterate is C(', K, ') =', C : 15 : 7);
          WRITELN;
          WRITELN('The current interval width is ', D : 15 : 7);
          WRITELN;
          WRITELN('The current function value  f(', C : 15 : 7, '  ) =', F(C) : 15 : 7);
          if Cond = 0 then
            begin
              WRITELN;
              WRITELN('The maximum number of iterations was exceeded.');
            end;
          if Cond = 4 then
            begin
              WRITELN;
              WRITELN('Surprise, a POLE was found instead of a ZERO!');
            end;
        end;
    end;
  end;                                      {End of PROCEDURE RESULTSB}

  procedure RESULTSF (A, B, C, DX: real; Delta, Epsilon: real;
                      K: integer; A0, A1, A2, B0, B1, B2, C0, C1, C2: real;
                      Satisfied: BOOLEAN);
    label
      999;
  begin
    CLRSCR;
    WRITELN('The false position method was used to find a zero of');
    WRITELN;
    WRITE('     ');
    PRINTFUNCTION(FunType);
    WRITELN;
    if YA * YB <= 0 then
      begin
        WRITELN('        Left point       Intersection point           Right point');
        WRITELN;
        WRITELN(A0 : 15 : 7, '        ', C0 : 15 : 7, '        ', B0 : 15 : 7);
        if K > 1 then
          WRITELN(A1 : 15 : 7, '        ', C1 : 15 : 7, '        ', B1 : 15 : 7);
        if K > 2 then
          WRITELN(A2 : 15 : 7, '        ', C2 : 15 : 7, '        ', B2 : 15 : 7);
      end;
    WRITELN;
    YA := F(A);
    YB := F(B);
    YC := F(C);
    if YA * YB > 0 then
      begin
        WRITELN('f(', A : 15 : 7, '  )  =', YA : 15 : 7);
        WRITELN;
        WRITELN('f(', B : 15 : 7, '  )  =', YB : 15 : 7);
        WRITELN;
        WRITELN('The values f(a) and f(b) do not differ in sign.');
        WRITELN;
        WRITELN('The false position method does not apply.');
        goto 999;
      end;
    if Satisfied = True then
      begin
        WRITELN('Convergence was successful.  After ', K : 3, ' iterations');
        WRITELN;
        WRITELN('The computed root of  f(x)  is   ', C : 15 : 7);
      end
    else
      begin
        WRITELN('Convergence was NOT successful.  After ', K : 3, ' iterations');
        WRITELN;
        WRITELN('The  current  iterate  is  c  =  ', C : 15 : 7);
      end;
    WRITELN;
    WRITELN('Consecutive iterates differ by   ', DX : 15 : 7);
    WRITELN;
    if DX < Delta then
      WRITELN('which is smaller than     Delta =', Delta : 15 : 7)
    else
      WRITELN(' which is LARGER than     Delta =', Delta : 15 : 7);
    WRITELN;
    WRITELN('       f(', C : 15 : 7, '  )  =', YC : 15 : 7);
    WRITELN;
    if ABS(YC) < Epsilon then
      WRITELN('which is smaller than   Epsilon =', Epsilon : 15 : 7)
    else
      WRITELN(' which is LARGER than   Epsilon =', Epsilon : 15 : 7);
    WRITELN;
    if YC = 0 then
      WRITELN('The computed function value is exactly zero!');
999:
  end;                                      {End of PROCEDURE RESULTSF}

  procedure PRINTAPPROXS;
    var
      J: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('  k             a                      c                      b  ');
    WRITELN('                 k                      k                      k ');
    WRITELN('--------------------------------------------------------------------------');
    WRITELN;
    for J := 0 to K do
      begin
        WRITELN(' ', J : 2, ' ', VA[J] : 18 : 7, '     ', VC[J] : 18 : 7, '     ', VB[J] : 18 : 7);
        WRITELN;
        if J mod 11 = 8 then
          begin
            WRITE('                    Press  the  <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
          end;
      end;
    WRITELN;
    WRITE('                    Press  the  <ENTER> key.  ');
    READLN(Ans);
  end;

begin                                            {Begin Main Program}
  Meth := 1;
  FunType := 1;
  A := 0;
  B := 1;
  Stat := Working;
  while (Stat = Working) do
    begin
      MESSAGE(FunType, Meth);
      State := Computing;
      while (State = Computing) do
        begin
          GETPOINTS(A, B, Delta, Epsilon);
          if Meth = 1 then
            begin
              BISECT(A, B, Delta, C, D, A0, A1, A2, B0, B1, B2, C0, C1, C2, Cond, K, KL, KR);
              RESULTSB(A, B, Delta, C, D, A0, A1, A2, B0, B1, B2, C0, C1, C2, Cond, K, KL, KR);
            end
          else
            begin
              FALSEPO(A, B, C, DX, Delta, Epsilon, K, A0, A1, A2, B0, B1, B2, C0, C1, C2, Satisfied);
              RESULTSF(A, B, C, DX, Delta, Epsilon, K, A0, A1, A2, B0, B1, B2, C0, C1, C2, Satisfied);
            end;
          WRITELN;
          WRITE('Do you want to see  all the approximations ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'Y') or (Ans = 'y') then
            PRINTAPPROXS;
          WRITELN;
          WRITE('Want  to try  a  different  interval [a,b] ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITE('Want to try a different function or method ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done
    end;
end.                                               {End Main Program}

